import random
from constants import *
from pygaze.display import Display
from pygaze.screen import Screen
from pygaze.keyboard import Keyboard
from pygaze.logfile import Logfile
import pygaze.libtime as timer

# create a Display to deal with the monitor
disp = Display()
# create a Keyboard to collect responses
kb = Keyboard(keylist=None, timeout=None)

# create a new log file
log = Logfile()
# define a header
header = ['fixonset', 'cueonset', 'cueoffset', 'taronset', \
    'cueside', 'tarside', 'valid', 'soa', 'target', \
    'response', 'correct', 'RT']
# write the header to the log file
log.write(header)

# define the instructions
instructions = 'Welcome!\n\nIn this experiment, Es and Fs \
will appear on either side of the screen. If you see \
an E, press the E key. If you see an F, press F. \
\n\nPlease try to be as fast and as accurate as \
possible.\n\nGood luck!'
# create a new Screen
instscr = Screen()
# draw the instructions on the Screen
instscr.draw_text(text=instructions, fontsize=24)

# create a new Screen
fixscr = Screen()
# draw a fixation cross in the centre
fixscr.draw_fixation(fixtype='cross', diameter=12)

# draw the left box
fixscr.draw_rect(x=BOXCORS['left'][0], y=BOXCORS['left'][1], \
    w=BOXSIZE, h=BOXSIZE, pw=3, fill=False)
# draw the right box
fixscr.draw_rect(x=BOXCORS['right'][0], y=BOXCORS['right'][1], \
    w=BOXSIZE, h=BOXSIZE, pw=3, fill=False)

# create a dict with two new Screens for the cues
cuescr = {}
cuescr['left'] = Screen()
cuescr['right'] = Screen()
# copy the fixation Screen to both cue Screens
cuescr['left'].copy(fixscr)
cuescr['right'].copy(fixscr)
# draw the cue boxes with thicker penwidths
cuescr['left'].draw_rect(x=BOXCORS['left'][0], y=BOXCORS['left'][1], \
    w=BOXSIZE, h=BOXSIZE, pw=8, fill=False)
cuescr['right'].draw_rect(x=BOXCORS['right'][0], y=BOXCORS['right'][1], \
    w=BOXSIZE, h=BOXSIZE, pw=8, fill=False)

# create a dict to contain further dicts to contain target Screens
tarscr = {}
tarscr['left'] = {}
tarscr['left']['E'] = Screen()
tarscr['left']['F'] = Screen()
tarscr['right'] = {}
tarscr['right']['E'] = Screen()
tarscr['right']['F'] = Screen()
# copy the fixation Screen to each target Screen
tarscr['left']['E'].copy(fixscr)
tarscr['left']['F'].copy(fixscr)
tarscr['right']['E'].copy(fixscr)
tarscr['right']['F'].copy(fixscr)
# calculate the target positions
tarpos = {}
tarpos['left'] = (BOXCORS['left'][0] + BOXSIZE/2, \
    BOXCORS['left'][1] + BOXSIZE/2)
tarpos['right'] = (BOXCORS['right'][0] + BOXSIZE/2, \
    BOXCORS['right'][1] + BOXSIZE/2)
# draw all possible targets on the target Screens
tarscr['left']['E'].draw_text(text='E', pos=tarpos['left'], fontsize=48)
tarscr['left']['F'].draw_text(text='F', pos=tarpos['left'], fontsize=48)
tarscr['right']['E'].draw_text(text='E', pos=tarpos['right'], fontsize=48)
tarscr['right']['F'].draw_text(text='F', pos=tarpos['right'], fontsize=48)

# create two new feedback Screens in a dict
fbscr = {}
# draw the incorrect feedback (evil red letters!)
fbscr[0] = Screen()
fbscr[0].draw_text(text='Incorrect!', colour=(255,0,0), fontsize=24)
# draw the correct feedback (nice and green)
fbscr[1] = Screen()
fbscr[1].draw_text(text='Correct!', colour=(0,255,0), fontsize=24)

# create an empty list to contain all unique trials
alltrials = []
# loop through all parameters
for cueside in CUELOCS:
    for tarside in TARLOCS:
        for soa in SOAS:
            for tar in TARGETS:
                # create a unique trial dict
                trial = {'cueside':cueside, 'tarside':tarside, \
                    'target':tar, 'soa':soa}
                # add the trial dict to the list
                alltrials.extend( TRIALREPEATS * [trial] )
# randomise the order of all trials
random.shuffle(alltrials)

# present the instructions
disp.fill(instscr)
disp.show()
# wait for any old keypress
kb.get_key(keylist=None, timeout=None)

# loop through all trials
for trial in alltrials:

    # show the fixation Screen
    disp.fill(fixscr)
    fixonset = disp.show()
    # wait for a bit
    timer.pause(FIXTIME)
    
    # show a cue Screen
    disp.fill(cuescr[trial['cueside']])
    cueonset = disp.show()
    # wait for a little bit
    timer.pause(CUETIME)
    
    # show the fixation Screen again
    disp.fill(fixscr)
    cueoffset = disp.show()
    # wait for the SOA minus the cue duration
    timer.pause(trial['soa'] - CUETIME - 10)
    
    # show a target Screen
    disp.fill(tarscr[trial['tarside']][trial['target']])
    taronset = disp.show()
    
    # wait for a response
    response, presstime = kb.get_key(keylist=['e', 'f'], timeout=None)
    # turn the lowercase response into uppercase
    response = response.upper()
    
    # check if the response was correct
    if response == trial['target']:
        correct = 1
    else:
        correct = 0
    
    # calculate the reaction time
    RT = presstime - taronset
    
    # check if the cue was valid
    if trial['cueside'] == trial['tarside']:
        validity = 1
    else:
        validity = 0
    
    # show the appropriate feedback Screen
    disp.fill(fbscr[correct])
    disp.show()
    # wait for a bit to allow the participant to see the feedback
    timer.pause(FEEDBACKTIME)

    # log all interesting values
    log.write([fixonset, cueonset, cueoffset, taronset, \
        trial['cueside'], trial['tarside'], validity, trial['soa'], \
        trial['target'], response, correct, RT])

# close the log file
log.close()

# shut down the experiment
disp.close()