import random
from constants import *
from psychopy.visual import Window, TextStim, Circle, Rect
from psychopy.event import waitKeys
from psychopy.core import wait

# create a Window to deal with the monitor
disp = Window(size=DISPSIZE, units='pix', \
    color=BGC, fullscr=True)

# open a new file instance
log = open(LOGFILE + '.tsv', 'w')
# define a header
header = ['fixonset', 'cueonset', 'cueoffset', 'taronset', \
    'cueside', 'tarside', 'valid', 'soa', 'target', \
    'response', 'correct', 'RT']
# make all values in the header into strings
# (all values are strings already, but this is an example)
line = map(str, header)
# join all string values into one string, separated by tabs ('\t')
line = '\t'.join(line)
# add a newline ('\n') to the string
line += '\n'
# write the header to the log file
log.write(line)

# define the instructions
instructions = 'Welcome!\n\nIn this experiment, Es and Fs \
will appear on either side of the screen. If you see \
an E, press the E key. If you see an F, press F. \
\n\nPlease try to be as fast and as accurate as \
possible.\n\nGood luck!'
# create a new text stimulus
inststim = TextStim(disp, text=instructions, color=FGC, height=24)

# create a Circle stimulus for fixation purposes
fixstim = Circle(disp, radius=6, edges=32, \
    lineColor=FGC, fillColor=FGC)

# create the left box
lboxstim = Rect(disp, pos=BOXCORS['left'], \
    width=BOXSIZE, height=BOXSIZE, lineColor=FGC, lineWidth=3)
# create the right box
rboxstim = Rect(disp, pos=BOXCORS['right'], \
    width=BOXSIZE, height=BOXSIZE, lineColor=FGC, lineWidth=3)

# create an empty dict to hold both cue stimuli
cuestim = {}
# create the left box
cuestim['left'] = Rect(disp, pos=BOXCORS['left'], \
    width=BOXSIZE, height=BOXSIZE, lineColor=FGC, lineWidth=8)
# create the right box
cuestim['right'] = Rect(disp, pos=BOXCORS['right'], \
    width=BOXSIZE, height=BOXSIZE, lineColor=FGC, lineWidth=8)

# create a dict to contain further dicts to contain target stimuli
tarstim = {}
tarstim['left'] = {}
tarstim['right'] = {}
# draw all possible target stimuli
tarstim['left']['E'] = TextStim(disp, text='E', pos=BOXCORS['left'],
    height=48, color=FGC)
tarstim['left']['F'] = TextStim(disp, text='F', pos=BOXCORS['left'],
    height=48, color=FGC)
tarstim['right']['E'] = TextStim(disp, text='E', pos=BOXCORS['right'],
    height=48, color=FGC)
tarstim['right']['F'] = TextStim(disp, text='F', pos=BOXCORS['right'],
    height=48, color=FGC)

# create a dict to hold two feedback stimuli
fbstim = {}
# draw the incorrect feedback (evil red letters!)
fbstim[0] = TextStim(disp, text='Incorrect!', height=24, \
    color=(1, -1, -1))
# draw the correct feedback (nice and green)
fbstim[1] = TextStim(disp, text='Correct!', height=24, \
    color=(-1, 1, -1))

# create an empty list to contain all unique trials
alltrials = []
# loop through all parameters
for cueside in CUELOCS:
    for tarside in TARLOCS:
        for soa in SOAS:
            for tar in TARGETS:
                # create a unique trial dict
                trial = {'cueside':cueside, 'tarside':tarside, \
                    'target':tar, 'soa':soa}
                # add the trial dict to the list
                alltrials.extend( TRIALREPEATS * [trial] )
# randomise the order of all trials
random.shuffle(alltrials)

# present the instructions
inststim.draw()
disp.flip()
# wait for any old keypress
waitKeys(maxWait=float('inf'), keyList=None, timeStamped=True)

# loop through all trials
for trial in alltrials:

    # draw the fixation mark, and the left and right boxes
    fixstim.draw()
    lboxstim.draw()
    rboxstim.draw()
    # update the monitor
    fixonset = disp.flip()
    # wait for a bit
    wait(FIXTIME)
    
    # draw the fixation mark, and the left and right boxes
    fixstim.draw()
    lboxstim.draw()
    rboxstim.draw()
    # draw a cue
    cuestim[trial['cueside']].draw()
    # update the monitor
    cueonset = disp.flip()
    # wait for a little bit
    wait(CUETIME)
    
    # draw the fixation mark, and the left and right boxes
    fixstim.draw()
    lboxstim.draw()
    rboxstim.draw()
    # update the monitor
    cueoffset = disp.flip()
    # wait for the SOA minus the cue duration
    wait(trial['soa'] - CUETIME - 0.01)
    
    # draw the fixation mark, and the left and right boxes
    fixstim.draw()
    lboxstim.draw()
    rboxstim.draw()
    # draw a target stimulus
    tarstim[trial['tarside']][trial['target']].draw()
    # update the monitor
    taronset = disp.flip()
    
    # wait for a response
    resplist = waitKeys(maxWait=float('inf'), keyList=['e','f'], \
        timeStamped=True)
    # select the first response from the response list
    response, presstime = resplist[0]
    # turn the lowercase response into uppercase
    response = response.upper()
    
    # check if the response was correct
    if response == trial['target']:
        correct = 1
    else:
        correct = 0
    
    # calculate the reaction time
    RT = presstime - taronset
    
    # check if the cue was valid
    if trial['cueside'] == trial['tarside']:
        validity = 1
    else:
        validity = 0
    
    # show the appropriate feedback stimulus
    fbstim[correct].draw()
    disp.flip()
    # wait for a bit to allow the participant to see the feedback
    wait(FEEDBACKTIME)

    # collect all interesting values in a single list
    line = [fixonset, cueonset, cueoffset, taronset, \
        trial['cueside'], trial['tarside'], validity, trial['soa'], \
        trial['target'], response, correct, RT]
    # turn all values into a string
    line = map(str, line)
    # merge all individual values into a single string, separated by tabs
    line = '\t'.join(line)
    # add a newline ('\n') to the string
    line += '\n'
    # write the data string to the log file
    log.write(line)

# close the log file
log.close()

# shut down the experiment
disp.close()
